/*
 * Decompiled with CFR 0.152.
 */
package com.sodiumarc.patchwork.app.scenecomposer.sound;

import com.sodiumarc.patchwork.util.MathUtils;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;
import org.apache.log4j.Logger;

public class AudioPlayer {
    public static final AudioFormat REQUIRED_FORMAT = new AudioFormat(AudioFormat.Encoding.PCM_SIGNED, 22050.0f, 16, 2, 4, 22050.0f, false);
    public static final int MAX_TRACKS = 16;
    public static final int LOOP_CONTINUOUSLY = -1;
    private SourceDataLine dataLine;
    private boolean active;
    private final Thread thread;
    private final List<Track> tracks = new ArrayList<Track>();
    private final Map<String, Track> tracksByID = new HashMap<String, Track>();
    protected int framePos;
    private static final Logger LOGGER = Logger.getLogger(AudioPlayer.class);

    public AudioPlayer() {
        this.thread = new Thread(){

            @Override
            public void run() {
                try {
                    AudioPlayer.this.run();
                }
                catch (LineUnavailableException e) {
                    e.printStackTrace();
                }
            }
        };
    }

    public synchronized void start() {
        if (this.active) {
            return;
        }
        this.active = true;
        this.thread.start();
    }

    public synchronized void stop() {
        this.active = false;
    }

    public synchronized boolean isActive() {
        return this.active;
    }

    public synchronized void playTrack(String id, byte[] soundData, AudioFormat format, int loopCount, float initialVolume, int fadeInMillis) {
        if (!REQUIRED_FORMAT.matches(format)) {
            throw new IllegalArgumentException("Incompatible audio format: " + format);
        }
        if (this.tracks.size() >= 16) {
            throw new IllegalArgumentException("Maximum track count exceeded.");
        }
        if (this.tracksByID.containsKey(id)) {
            this.removeTrack(id);
        }
        Track track = new Track(id, soundData);
        track.start(loopCount, initialVolume, fadeInMillis);
        this.tracksByID.put(id, track);
        this.tracks.add(track);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Playing Clip: " + id);
        }
        this.thread.interrupt();
    }

    public Set<String> getTrackIDs() {
        return this.tracksByID.keySet();
    }

    public synchronized int getTrackCount() {
        return this.tracks.size();
    }

    public synchronized void stopTrack(String id, int fadeOutMillis) {
        Track track = this.tracksByID.get(id);
        if (track != null) {
            track.stop(fadeOutMillis);
        }
    }

    public synchronized void stopAllTracks() {
        this.tracksByID.clear();
        this.tracks.clear();
    }

    public synchronized boolean isPlaying() {
        return !this.tracks.isEmpty();
    }

    public synchronized boolean isTrackPlaying(String id) {
        return !this.tracksByID.containsKey(id);
    }

    public synchronized void setTrackVolume(String id, float volume) {
        Track track = this.tracksByID.get(id);
        if (track != null) {
            track.setVolume(volume);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void run() throws LineUnavailableException {
        this.dataLine = AudioSystem.getSourceDataLine(REQUIRED_FORMAT);
        this.dataLine.open(REQUIRED_FORMAT);
        this.dataLine.start();
        int frameSize = REQUIRED_FORMAT.getFrameSize();
        int dataLineBufferFrames = this.dataLine.getBufferSize() / frameSize;
        int transferBufferFrames = dataLineBufferFrames / 4;
        int transferBufferSamples = transferBufferFrames * REQUIRED_FORMAT.getChannels();
        int transferBufferBytes = transferBufferFrames * frameSize;
        int[][] trackBuffers = new int[16][transferBufferSamples];
        int[] trackSamplesRead = new int[16];
        byte[] mixBuffer = new byte[transferBufferBytes];
        while (true) {
            int trackCount = 0;
            boolean active = false;
            AudioPlayer audioPlayer = this;
            synchronized (audioPlayer) {
                active = this.active;
                trackCount = this.tracks.size();
            }
            if (!active) break;
            if (trackCount > 0) {
                int available = this.dataLine.available();
                if (available >= transferBufferBytes) {
                    int maxSamplesRead = 0;
                    int finishedTrackCount = 0;
                    AudioPlayer audioPlayer2 = this;
                    synchronized (audioPlayer2) {
                        for (int i = 0; i < trackCount; ++i) {
                            int samplesRead = this.tracks.get(i).read(trackBuffers[i]);
                            if ((maxSamplesRead = Math.max(maxSamplesRead, samplesRead)) == 0) {
                                ++finishedTrackCount;
                            }
                            trackSamplesRead[i] = samplesRead;
                        }
                    }
                    if (maxSamplesRead > 0) {
                        this.mix(trackBuffers, mixBuffer, trackCount, trackSamplesRead);
                        int bytesWritten = this.dataLine.write(mixBuffer, 0, maxSamplesRead * 2);
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("Wrote " + bytesWritten + " audio bytes.");
                        }
                        this.framePos += bytesWritten / frameSize;
                    }
                    if (finishedTrackCount > 0) {
                        AudioPlayer audioPlayer3 = this;
                        synchronized (audioPlayer3) {
                            HashSet<String> finishedTrackIds = new HashSet<String>();
                            for (int i = 0; i < trackCount; ++i) {
                                if (trackSamplesRead[i] != 0) continue;
                                finishedTrackIds.add(this.tracks.get(i).getId());
                            }
                            for (String id : finishedTrackIds) {
                                this.removeTrack(id);
                                if (!LOGGER.isDebugEnabled()) continue;
                                LOGGER.debug("Retiring track " + id);
                            }
                        }
                    }
                }
                if (this.framePos == 0 || this.dataLine.getBufferSize() != this.dataLine.available()) continue;
                System.err.println("Buffer underrun!");
                continue;
            }
            try {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Sleeping...");
                }
                Thread.sleep(500L);
            }
            catch (InterruptedException e) {}
        }
        this.dataLine.drain();
        this.dataLine.stop();
        this.dataLine.close();
        this.dataLine = null;
    }

    private synchronized void removeTrack(String id) {
        Track track = this.tracksByID.remove(id);
        if (track != null) {
            this.tracks.remove(track);
        }
    }

    private void mix(int[][] tracks, byte[] destination, int trackCount, int[] trackLengths) {
        int frameSize = REQUIRED_FORMAT.getFrameSize();
        int channels = REQUIRED_FORMAT.getChannels();
        int bytesPerSample = frameSize / channels;
        int trackLen = tracks[0].length;
        int sampleIndex = 0;
        int byteIndex = 0;
        while (sampleIndex < trackLen) {
            int mixedSample = 0;
            for (int t = 0; t < trackCount; ++t) {
                int[] track = tracks[t];
                if (sampleIndex >= trackLengths[t]) continue;
                mixedSample += track[sampleIndex];
            }
            AudioPlayer.int16ToBytes(mixedSample, destination, byteIndex);
            ++sampleIndex;
            byteIndex += bytesPerSample;
        }
    }

    private static int bytesToInt16(byte[] bytes, int offset) {
        return bytes[offset + 0] & 0xFF | bytes[offset + 1] << 8;
    }

    private static void int16ToBytes(int sample, byte[] bytes, int offset) {
        bytes[offset + 0] = (byte)(sample & 0xFF);
        bytes[offset + 1] = (byte)(sample >> 8 & 0xFF);
    }

    private static class Track {
        private final String id;
        private final byte[] soundData;
        private final int[] soundDataInt;
        private final int clipFrameLength;
        private long currentFrame;
        private long endFrame;
        private float volume = 1.0f;
        private int decibelsInv;
        private int volumeDecibels;
        private int[] fadeFrames;
        private float[] fadeVolumes;
        private int fadeIndex = -1;
        private static Logger LOGGER = Logger.getLogger(Track.class);
        private static final int VOLUME_RESOLUTION = 100;

        public Track(String id, byte[] soundData) {
            this.id = id;
            this.soundData = soundData;
            int frameSize = REQUIRED_FORMAT.getFrameSize();
            int channels = REQUIRED_FORMAT.getChannels();
            int bytesPerSample = frameSize / channels;
            this.soundDataInt = new int[soundData.length / bytesPerSample];
            int i = 0;
            int bytePos = 0;
            while (i < this.soundDataInt.length) {
                this.soundDataInt[i] = 100 * AudioPlayer.bytesToInt16(soundData, bytePos);
                ++i;
                bytePos += 2;
            }
            this.clipFrameLength = soundData.length / REQUIRED_FORMAT.getFrameSize();
        }

        public int read(int[] destination) {
            if (!this.isContinuous() && this.getFramesRemaining() == 0L) {
                return 0;
            }
            int channels = REQUIRED_FORMAT.getChannels();
            int samplePos = (int)(this.currentFrame % (long)this.clipFrameLength) * channels;
            int samplesRead = this.getIntData(this.soundDataInt, destination, samplePos, this.currentFrame, true);
            if (!this.isContinuous()) {
                samplesRead = (int)Math.min((long)samplesRead, this.getFramesRemaining() * (long)channels);
            }
            this.currentFrame += (long)(samplesRead / channels);
            if (!this.isContinuous()) {
                this.currentFrame = Math.min(this.currentFrame, this.endFrame + 1L);
            }
            return samplesRead;
        }

        public String getId() {
            return this.id;
        }

        public boolean isContinuous() {
            return this.endFrame == -1L;
        }

        public long getFramesRemaining() {
            if (this.isContinuous()) {
                return -1L;
            }
            return this.endFrame - this.currentFrame + 1L;
        }

        public void start(int loopsRemaining, float initialVolume, int fadeInMillis) {
            this.currentFrame = 0L;
            this.setLoopsRemaining(loopsRemaining);
            this.setVolume(0.0f);
            this.fadeToVolume(initialVolume, fadeInMillis);
        }

        public void setLoopsRemaining(int loopsRemaining) {
            this.endFrame = loopsRemaining < 0 ? -1L : this.currentFrame + (long)(loopsRemaining * this.clipFrameLength) - 1L;
        }

        public float getVolume() {
            return this.volume;
        }

        public void stop(int fadeOutMillis) {
            int fadeDurationFrames = (int)(REQUIRED_FORMAT.getFrameRate() / 1000.0f * (float)fadeOutMillis);
            this.endFrame = this.currentFrame + (long)fadeDurationFrames;
            this.fadeToVolume(0.0f, fadeOutMillis);
        }

        public void setVolume(float volume) {
            this.volume = MathUtils.clamp(volume, 0.0f, 1.0f);
            this.decibelsInv = this.getDecibelsInv(volume);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Setting volume = " + volume + ", denom = " + this.decibelsInv);
            }
        }

        public void fadeToVolume(float targetVolume, int fadeDurationMillis) {
            if (fadeDurationMillis == 0) {
                this.setVolume(targetVolume);
                return;
            }
            targetVolume = MathUtils.clamp(targetVolume, 0.0f, 1.0f);
            int fadeDurationFrames = (int)(REQUIRED_FORMAT.getFrameRate() / 1000.0f * (float)fadeDurationMillis);
            long fadeStartFrame = this.currentFrame;
            long fadeEndFrame = fadeStartFrame + (long)fadeDurationFrames;
            this.fadeToVolume(targetVolume, fadeStartFrame, fadeEndFrame);
        }

        private int getByteData(byte[] source, byte[] destination, int sourceStartIndex, boolean wrap) {
            int destIndex;
            int sourceIndex = sourceStartIndex;
            for (destIndex = 0; destIndex < destination.length; ++destIndex) {
                destination[destIndex] = source[sourceIndex];
                if (++sourceIndex != source.length) continue;
                if (!wrap) break;
                sourceIndex = 0;
            }
            return destIndex;
        }

        private int getIntData(int[] source, int[] destination, int sourceStartIndex, long startFrame, boolean wrap) {
            int destIndex;
            int sourceIndex = sourceStartIndex;
            long frame = startFrame;
            for (destIndex = 0; destIndex < destination.length; ++destIndex) {
                int sample = source[sourceIndex];
                if (sourceIndex % 2 == 1) {
                    this.updateDecibelsInv(++frame);
                }
                destination[destIndex] = this.volumeAdjust(sample);
                if (++sourceIndex != source.length) continue;
                if (!wrap) break;
                sourceIndex = 0;
            }
            return destIndex;
        }

        private void fadeToVolume(float targetVolume, long fadeStartFrame, long fadeEndFrame) {
            this.fadeIndex = -1;
            this.fadeFrames = null;
            this.fadeVolumes = null;
            if (fadeStartFrame == fadeEndFrame) {
                this.setVolume(targetVolume);
                return;
            }
            targetVolume = MathUtils.clamp(targetVolume, 0.0f, 1.0f);
            int fadeSteps = (int)(Math.abs(targetVolume - this.volume) * 100.0f);
            this.fadeFrames = new int[fadeSteps];
            this.fadeVolumes = new float[fadeSteps];
            for (int i = 0; i < fadeSteps; ++i) {
                float fadeFraction = (float)i / (float)(fadeSteps - 1);
                this.fadeFrames[i] = (int)MathUtils.interpolate(fadeFraction, fadeStartFrame, fadeEndFrame);
                this.fadeVolumes[i] = (float)MathUtils.interpolate(fadeFraction, this.volume, targetVolume);
            }
            this.fadeIndex = 0;
        }

        private int getDecibelsInv(float volume) {
            int result;
            if (volume == 0.0f) {
                result = 0;
            } else if (volume == 1.0f) {
                result = 100;
            } else {
                float db1 = (float)(1.0 / MathUtils.rangeFraction(Math.pow(10.0, volume), 1.0, 10.0));
                float db2 = (float)(1.0 + 20.0 * Math.log10(1.0f / volume));
                result = Math.round(100.0f * db1);
            }
            return result;
        }

        private void updateDecibelsInv(long frame) {
            if (this.fadeIndex >= 0 && this.fadeIndex < this.fadeFrames.length - 1 && frame > (long)this.fadeFrames[this.fadeIndex + 1]) {
                ++this.fadeIndex;
                this.setVolume(this.fadeVolumes[this.fadeIndex]);
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Updating fade volume to " + this.decibelsInv);
                }
                if (this.fadeIndex >= this.fadeFrames.length - 1) {
                    this.fadeIndex = -1;
                    this.fadeFrames = null;
                    this.fadeVolumes = null;
                }
            }
        }

        private int volumeAdjust(int sample) {
            if (this.decibelsInv == 0) {
                return 0;
            }
            return sample / this.decibelsInv;
        }
    }
}

